<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Curso;
use App\Models\Tarea;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\CursoDocente;
use App\Models\SubcomponenteLogro;
use App\Models\Sesion;
class TareaController extends Controller
{
    /**
     * Mostrar el listado de tareas de un curso.
     */
   public function index($curso_id)
{
    $docente_id = Auth::user()->docente->id ?? null;

    // Validar que el curso pertenece al docente
    $asignado = CursoDocente::where('curso_id', $curso_id)
        ->where('docente_id', $docente_id)
        ->exists();

    if (!$asignado) {
        abort(403, 'No tienes permiso para acceder a este curso.');
    }

    // Curso
    $curso = Curso::findOrFail($curso_id);

    // TAREAS
    $tareas = Tarea::where('curso_id', $curso->id)
                    ->orderBy('fecha', 'asc')
                    ->orderBy('hora_inicio', 'asc')
                    ->get();

    // 🔹 SUBCOMPONENTES DEL CURSO (solo los que pertenecen a este curso)
    $subcomponentes = SubcomponenteLogro::whereHas('logroCurso', function ($q) use ($curso) {
        $q->where('curso_id', $curso->id);
    })->get();

    // 🔹 SESIONES DEL CURSO
    $sesiones = Sesion::where('curso_id', $curso->id)->get();

    return view('docentes.tareas.index', compact(
        'curso',
        'tareas',
        'subcomponentes',
        'sesiones'
    ));
}

    /**
     * Registrar una tarea nueva.
     */
   public function store(Request $request, $curso_id)
{
    $curso = Curso::findOrFail($curso_id);

    $request->validate([
        'nombre' => 'required|max:100',
        'descripcion' => 'nullable|string',
        'fecha' => 'nullable|date',
        'fecha_fin' => 'nullable|date',
        'hora_inicio' => 'nullable',
        'hora_fin' => 'nullable',
        'subcomponente_id' => 'nullable|exists:subcomponentes_logro,id',
        'sesion_id' => 'nullable|exists:sesiones,id_sesion',
        'archivo' => 'nullable|file|max:5120', // max 5MB
    ]);

    $archivoNombre = null;

    // 🔹 Crear carpeta si no existe
    $rutaCarpeta = public_path('tareas');
    if (!file_exists($rutaCarpeta)) {
        mkdir($rutaCarpeta, 0777, true);
    }

    // 🔹 Si se subió archivo, lo guardamos
    if ($request->hasFile('archivo')) {
        $archivo = $request->file('archivo');
        
        // Nombre único
        $archivoNombre = time() . '_' . uniqid() . '.' . $archivo->getClientOriginalExtension();
        
        // Mover archivo a public/tareas
        $archivo->move($rutaCarpeta, $archivoNombre);
    }

    // 🔹 Crear tarea
    Tarea::create([
        'curso_id' => $curso->id,
        'nombre' => $request->nombre,
        'descripcion' => $request->descripcion,
        'fecha' => $request->fecha,
        'fecha_fin' => $request->fecha_fin,
        'hora_inicio' => $request->hora_inicio,
        'hora_fin' => $request->hora_fin,
        'subcomponente_id' => $request->subcomponente_id,
        'sesion_id' => $request->sesion_id,
        'archivo' => $archivoNombre
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Tarea registrada correctamente.'
    ]);
}


    /**
     * Actualizar una tarea específica.
     */
    public function update(Request $request, $id)
{
    $tarea = Tarea::findOrFail($id);

    $request->validate([
        'nombre' => 'required|max:100',
        'descripcion' => 'nullable|string',
        'fecha' => 'nullable|date',
        'fecha_fin' => 'nullable|date',
        'hora_inicio' => 'nullable',
        'hora_fin' => 'nullable',
        'subcomponente_id' => 'nullable|exists:subcomponentes_logro,id',
        'sesion_id' => 'nullable|exists:sesiones,id_sesion',
        'archivo' => 'nullable|file|max:5120', // 5MB
    ]);

    $archivoNombre = $tarea->archivo;

    // Si envía un archivo nuevo → reemplazar
    if ($request->hasFile('archivo')) {

        // Crear carpeta si no existe
        $rutaCarpeta = public_path('tareas');
        if (!file_exists($rutaCarpeta)) {
            mkdir($rutaCarpeta, 0777, true);
        }

        // Borrar archivo anterior
        if ($tarea->archivo && file_exists($rutaCarpeta . '/' . $tarea->archivo)) {
            unlink($rutaCarpeta . '/' . $tarea->archivo);
        }

        // Subir nuevo archivo
        $archivo = $request->file('archivo');
        $archivoNombre = time() . '_' . uniqid() . '.' . $archivo->getClientOriginalExtension();
        $archivo->move($rutaCarpeta, $archivoNombre);
    }

    $tarea->update([
        'nombre' => $request->nombre,
        'descripcion' => $request->descripcion,
        'fecha' => $request->fecha,
        'fecha_fin' => $request->fecha_fin,
        'hora_inicio' => $request->hora_inicio,
        'hora_fin' => $request->hora_fin,
        'subcomponente_id' => $request->subcomponente_id,
        'sesion_id' => $request->sesion_id,
        'archivo' => $archivoNombre,
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Tarea actualizada correctamente.'
    ]);
}


    /**
     * Eliminar tarea.
     */
    public function destroy($id)
{
    $tarea = Tarea::findOrFail($id);

    // Eliminar archivo si existe
    if ($tarea->archivo && file_exists(public_path('tareas/' . $tarea->archivo))) {
        unlink(public_path('tareas/' . $tarea->archivo));
    }

    $tarea->delete();

    return response()->json([
        'success' => true,
        'message' => 'Tarea eliminada correctamente.'
    ]);
}

}
