<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Postulante;
use App\Models\Inscripcion;
use App\Models\Admision;
use App\Models\PlanEstudio;
use App\Models\Instituto;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\InscripcionesEstadoExport;
class InscripcionController extends Controller
{
    // Mostrar formulario público
    public function create()
    {
        $instituto = Instituto::first(); // datos institucionales

        // Solo admisiones activas
        $admisiones = Admision::where('activo', 1)->get();

        // Planes activos
        $planes = PlanEstudio::where('activo', 1)->get();

        return view('postulacion.form', compact('instituto', 'admisiones', 'planes'));
    }


    // Guardar inscripción pública
   public function store(Request $request)
{
    // ========================================
    // VALIDACIÓN COMPLETA
    // ========================================
    $request->validate([
        'nombres'          => 'required|string|max:100',
        'apellidos'        => 'required|string|max:100',

        // VALIDAR QUE EL DNI NO EXISTA EN POSTULANTES NI USUARIOS
        'dni'              => [
            'required',
            'string',
            'max:15',
            'unique:postulantes,dni'
            
        ],

        'email'            => 'required|email',
        'telefono'         => 'nullable|string|max:15',
        'genero'           => 'required',
        'fecha_nacimiento' => 'required|date',
        'direccion'        => 'nullable|string|max:255',

        'id_admision'      => 'required|exists:admisiones,id_admision',
        'id_plan'          => 'required|exists:planes_estudio,id',

        // ARCHIVOS OBLIGATORIOS
        'foto_postulante'     => 'required|image|mimes:jpg,jpeg,png|max:2048',
        'certificado_estudios'=> 'required|mimes:pdf|max:4096',
        'partida_nacimiento'  => 'required|mimes:pdf|max:4096',
        'copia_dni'           => 'required|mimes:pdf,jpg,jpeg,png|max:4096',
        'comprobante_pago'    => 'required|mimes:pdf,jpg,jpeg,png|max:4096',
    ]);

    // ========================================
    // CARPETA DENTRO DE PUBLIC/
    // ========================================
    $carpeta = public_path('documentospostulantes/' . $request->dni);

    if (!file_exists($carpeta)) {
        mkdir($carpeta, 0755, true);
    }

    // ========================================
    // GUARDAR ARCHIVOS EN /public/documentospostulantes/dni/
    // ========================================
    $guardarArchivo = function($file, $nombre) use ($carpeta) {

        $extension = $file->getClientOriginalExtension();
        $filename = $nombre . '_' . time() . '.' . $extension;

        $file->move($carpeta, $filename);

        // Ruta que se almacena en BD
        return 'documentospostulantes/' . request()->dni . '/' . $filename;
    };

    $foto        = $guardarArchivo($request->file('foto_postulante'), 'foto');
    $certificado = $guardarArchivo($request->file('certificado_estudios'), 'certificado');
    $partida     = $guardarArchivo($request->file('partida_nacimiento'), 'partida');
    $dni         = $guardarArchivo($request->file('copia_dni'), 'dni');
    $pago        = $guardarArchivo($request->file('comprobante_pago'), 'pago');


    // ========================================
    // REGISTRO POSTULANTE
    // ========================================
    $postulante = Postulante::create([
        'nombres'              => $request->nombres,
        'apellidos'            => $request->apellidos,
        'dni'                  => $request->dni,
        'email'                => $request->email,
        'telefono'             => $request->telefono,
        'genero'               => $request->genero,
        'fecha_nacimiento'     => $request->fecha_nacimiento,
        'direccion'            => $request->direccion,

        'foto_postulante'      => $foto,
        'certificado_estudios' => $certificado,
        'partida_nacimiento'   => $partida,
        'copia_dni'            => $dni,
        'comprobante_pago'     => $pago,

        // 🔥 Asignar automáticamente grado
        'grado'                => 'Postulante',
    ]);

    // ========================================
    // REGISTRO INSCRIPCIÓN
    // ========================================
    Inscripcion::create([
        'id_admision'    => $request->id_admision,
        'id_postulante'  => $postulante->id_postulante,
        'id_plan'        => $request->id_plan,
        'segunda_opcion' => $request->segunda_opcion,
        'estado'         => 'inscrito',
    ]);

    return redirect()
        ->route('postulacion.form')
        ->with('success', '¡Inscripción registrada correctamente! Su código es: ' . $postulante->id_postulante);
}

public function index()
{
    $inscripciones = Inscripcion::with(['postulante', 'admision', 'plan'])
        ->where('estado', 'inscrito')
        ->get();

    return view('inscripciones.index', compact('inscripciones'));
}

public function aceptadas()
{
    $inscripciones = Inscripcion::with(['postulante', 'admision', 'plan'])
        ->where('estado', 'aceptado')
        ->get();

    return view('inscripciones.aceptadas', compact('inscripciones'));
}

public function observadas()
{
    $inscripciones = Inscripcion::with(['postulante', 'admision', 'plan'])
        ->where('estado', 'observado')
        ->get();

    return view('inscripciones.observadas', compact('inscripciones'));
}

public function rechazadas()
{
    $inscripciones = Inscripcion::with(['postulante', 'admision', 'plan'])
        ->where('estado', 'rechazado')
        ->get();

    return view('inscripciones.rechazadas', compact('inscripciones'));
}

public function cambiarEstado(Request $request, $id)
{
    $request->validate([
        'estado' => 'required|in:inscrito,observado,subsanado,aceptado,matriculado'
    ]);

    $inscripcion = Inscripcion::findOrFail($id);
    $inscripcion->estado = $request->estado;
    $inscripcion->observacion = $request->observacion;
    $inscripcion->save();

    return response()->json(['success' => true]);
}

public function exportarAceptados()
{
    return Excel::download(new InscripcionesEstadoExport('aceptado'), 'aceptados.xlsx');
}
}
