<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Aula;
use App\Models\Pabellon;

class AulaController extends Controller
{
    /**
     * Mostrar listado de aulas
     */
    public function index()
    {
        $aulas = Aula::with('pabellon')->orderBy('id', 'desc')->get();
        $pabellones = Pabellon::orderBy('nombre')->get();

        return view('aulas.index', compact('aulas', 'pabellones'));
    }

    /**
     * Registrar nueva aula (AJAX)
     */
    public function store(Request $request)
{
    $request->validate([
        'nombre'        => 'required|string|max:100',
        'numero_aula'   => 'nullable|string|max:20',
        'capacidad'     => 'nullable|integer|min:1',
        'id_pabellon'   => 'required|exists:pabellones,id',
        'tipo'          => 'nullable|string|max:50',
    ]);

    Aula::create([
        'nombre'        => $request->nombre,
        'numero_aula'   => $request->numero_aula,
        'capacidad'     => $request->capacidad,
        'id_pabellon'   => $request->id_pabellon,
        'tipo'          => $request->tipo,
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Aula registrada correctamente.'
    ]);
}


    /**
     * Actualizar aula (AJAX)
     */
    public function update(Request $request, $id)
{
    $aula = Aula::findOrFail($id);

    $request->validate([
        'nombre'        => 'required|string|max:100',
        'numero_aula'   => 'nullable|string|max:50',
        'capacidad'     => 'nullable|integer|min:1',
        'id_pabellon'   => 'required|exists:pabellones,id',
        'tipo'          => 'required|string|in:Aula,Laboratorio',
    ]);

    try {

        $aula->update([
            'nombre'        => $request->nombre,
            'numero_aula'   => $request->numero_aula,
            'capacidad'     => $request->capacidad,
            'id_pabellon'   => $request->id_pabellon,
            'tipo'          => $request->tipo,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Aula actualizada correctamente.'
        ]);

    } catch (\Exception $e) {

        return response()->json([
            'success' => false,
            'message' => 'Ocurrió un error al actualizar el aula.',
            'error'   => $e->getMessage()
        ], 500);

    }
}

public function show($id)
{
    $aula = Aula::with('pabellon')->find($id);

    if (!$aula) {
        return response()->json([
            'success' => false,
            'message' => 'Aula no encontrada.'
        ], 404);
    }

    return response()->json([
        'success' => true,
        'data' => [
            'id'           => $aula->id,
            'nombre'       => $aula->nombre,
            'numero_aula'  => $aula->numero_aula,
            'capacidad'    => $aula->capacidad,
            'tipo'         => $aula->tipo,
            'pabellon'     => $aula->pabellon->nombre ?? null,
        ]
    ]);
}

    /**
     * Eliminar aula (AJAX)
     */
    public function destroy($id)
    {
        $aula = Aula::findOrFail($id);

        // Opcional: validar si tiene horarios asignados
        // if ($aula->horarios()->exists()) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'El aula está asignada a un horario y no puede eliminarse.'
        //     ]);
        // }

        $aula->delete();

        return response()->json([
            'success' => true,
            'message' => 'Aula eliminada correctamente.'
        ]);
    }
}
